"use client"

import { useCallback, useEffect, useRef, useState } from 'react';

import { addBlockAction, Block, deleteBlockAction, updateBlockAction } from '@actions';

export default function DocumentBlock({ block, setContextMenuCoordinates, setFetchData, setFocusedBlock }: {
  block?: Block,
  setContextMenuCoordinates?: React.Dispatch<React.SetStateAction<{ x: number; y: number } | undefined>>,
  setFetchData: React.Dispatch<React.SetStateAction<boolean>>,
  setFocusedBlock?: React.Dispatch<React.SetStateAction<Block | undefined>>,
}) {
  const [content, setContent] = useState(block?.content || '');
  const blockRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    if (!blockRef.current) return;
    if (block?.type === 'img') {
      blockRef.current.innerHTML = `<img class="h-auto max-w-md" src="${block.content}">`;
    } else {
      blockRef.current.textContent = content;
    }
  }, [block, content]);

  const handleInput = useCallback((e: React.ChangeEvent<HTMLInputElement>) => {
    if (typeof e.currentTarget.textContent === 'string') {
      setContent(e.currentTarget.textContent);
    }
  }, []);

  const handleBlurInput = useCallback(() => {
    if (content.length) {
      const type = content.includes('https://') ? 'img' : 'p';
      addBlockAction({ type, content });
      setFetchData(true);
      if (blockRef.current) {
        blockRef.current.textContent = '';
      }
      setContent('');
    }
  }, [content, setContent, setFetchData]);

  const handleBlurBlock = useCallback(() => {
    setContextMenuCoordinates?.(undefined);
    if (content.length) {
      if (content !== block?.content) {
        updateBlockAction({ ...block, content: content } as Block);
      }
    } else if (block?.id) {
      deleteBlockAction(block.id);
      setFetchData(true);
    }
  }, [block, content, setContextMenuCoordinates, setFetchData]);

  const handleKeyDown = useCallback((e: React.KeyboardEvent<HTMLInputElement>) => {
    if (e.key === 'Enter') {
      e.preventDefault();
      handleBlurInput();
    }
  }, [handleBlurInput]);

  const handleContextMenu = useCallback((e: React.MouseEvent<HTMLInputElement>) => {
    e.preventDefault();
    setFocusedBlock?.(block);
    setContextMenuCoordinates?.({
      x: e.pageX,
      y: e.pageY,
    });
  }, [block, setContextMenuCoordinates, setFocusedBlock]);

  if (!block) {
    return (
      <div className="grid">
        {content.length > 0 || <div className="col-start-1 p-2 row-start-1 text-gray-600">Write something...</div>}
        <div className="col-start-1 p-2 row-start-1 z-10" contentEditable suppressContentEditableWarning onInput={handleInput} onBlur={handleBlurInput} onKeyDown={handleKeyDown} ref={blockRef}></div>
      </div>
    );
  }

  const classes = ['p-2'];
  classes.push('[&[contenteditable]]:focus:border-none [&[contenteditable]]:focus:outline-none [&[contenteditable]]:active:border-none [&[contenteditable]]:active:outline-none');

  switch (block.type) {
    case 'h1':
      classes.push('text-4xl');
      break;
    case 'h2':
      classes.push('text-3xl');
      break;
    case 'h3':
      classes.push('text-2xl');
      break;
  }

  const contentEditable = block.type !== 'img';

  return (
    <div className={classes.join(' ')} contentEditable={contentEditable} suppressContentEditableWarning onInput={handleInput} onBlur={handleBlurBlock} ref={blockRef} onContextMenu={handleContextMenu}></div>
  );
}
